;*********************************************************************
;*
;*              SRAM (16-bit) Routines for PICDEM-18R Demo board
;*
;*********************************************************************
;* FileName:        SRAM16.ASM
;* Dependencies:    PIC18C601/801.inc
;* Processor:       PIC18C601/801
;* Complier:        MPLAB 5.11.00
;* Company:         Microchip Technology, Inc.
;*
;* Software License Agreement
;*
;* The software supplied herewith by Microchip Technology Incorporated
;* (the Company) for its PICmicro Microcontroller is intended and
;* supplied to you, the Companys customer, for use solely and
;* exclusively on Microchip PICmicro Microcontroller products. The
;* software is owned by the Company and/or its supplier, and is
;* protected under applicable copyright laws. All rights are reserved.
;* Any use in violation of the foregoing restrictions may subject the
;* user to criminal sanctions under applicable laws, as well as to
;* civil liability for the breach of the terms and conditions of this
;* license.
;*
;* THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
;* WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
;* TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
;* PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
;* IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
;* CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
;*
;*
;* NOTE: This file must only use "bra" and "rcall" jump instructions.
;*       This code is "relocated" at run time and hence this
;*       restriction must be ovserved.
;*
;* Author               Date        Comment
;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;* NKR                  3/5/01     Original (Rev. 1.0)
;********************************************************************/

#include <p18c801.inc>

#include "MEMRTNES.INC"


SWAP_CS_LINES       EQU     02h
RESTORE_CS_LINES    EQU     00h

; Just a helpful macro to unlock and modify secured registers.
UNLOCK_N_MODIFY macro   @REG
    bsf     PSPCON, CMLK1                   ; 
    bsf     PSPCON, CMLK0
    movwf   @REG
    endm

; Any local/private data this programmer may have should go here.
ProgrammerData      UDATA_ACS
BufferedLSB         RES     01h


Programmer          CODE   1000h

JumpTable:
; Jump table for Memory routines - used by monitor/bootloader code.
    bra     Write           
    bra     Erase           





;***************************************************************************
; Function: WREG Write(DWORD Address, BYTE Byte, BYTE Flags)
;
; Input:    Address                         : 32-bit address to write to
;           Byte                            ; 8-bit data value to be written
;           Flags                           ; Special flags
;                               
;
; Output:   WERG                            : 0 if successful
;                                           ; Else error code
;
; Side Effects: Databank is changed
;
; Overview: 
;***************************************************************************
Write:
    btfsc   Address, 0                      ; Is this odd address ?
    bra     Write_Continue
    movff   Byte, BufferedLSB               ; Remember LSB until we receive MSB
    retlw   00h

Write_Continue:
    rcall   _InitializeProgrammer           ; Setup appropriate Table Write mode
                                            ; This could be done only once based
                                            ; on flag.  We will do it later on..

    movlw   SWAP_CS_LINES
    rcall   SwapCS

    movff   Address, TBLPTRL                ; Assume SRAM at this address
    bcf     TBLPTRL, 0                      ; Word align it.
    movff   Address+1, TBLPTRH              ; and attempt to write it.
    movff   Address+2, TBLPTRU            
    movff   BufferedLSB, TABLAT

    tblwt*+                                 ; Write LSB

    movff   Byte, TABLAT                    ; Write MSB
    tblwt*

    tblrd*-                                 ; Verify MSB part. 

    movf    Byte, W                        
    xorwf   TABLAT, W                     
    bnz     Write_Error                     

    tblrd*                                  ; Verify LSB part.

    movf    BufferedLSB, W
    xorwf   TABLAT, W
    bnz     Write_Error

Write_Error:
    movwf   PRODL                           ; Save operation result.
    movf    PRODL, W                        ; Restore operation result.
    return                              



;***************************************************************************
; Function: WREG Erase(void)
;
; Input:    None
;
; Output:   WERG                            : 0 if successful
;                                           ; Else error code
;
; Side Effects: Databank is changed
;
; Overview: 
;***************************************************************************
Erase:
    retlw   00h                             ; Nothing to erase on SRAM.




;***************************************************************************
; Function: void _InitializeProgrammer()
;
; Input:    None
;
; Output:   None
;
; Side Effects: Databank is changed
;
; Overview: Initialize all peripherals on board.
;***************************************************************************
_InitializeProgrammer:
    movf    MEMCON, W                       ; Set 16-bit Byte Write mode.
    bsf     WREG, WM1                       ; Select 16-bit Write Mode
    bcf     WREG, WM0
    UNLOCK_N_MODIFY MEMCON
    return



;***************************************************************************
; Function: void SwapCS(WREG)
;
; Input:    WREG
;
; Output:   None
;
; Side Effects: Databank is changed
;
; Overview: Load given value directly in TABLAT
;           Setup CSIO access
;           Write TABLAT to CSIO.0FEh address to program CPLD
;***************************************************************************
SwapCS:
    movwf   TABLAT                          ; Load given value

    movlw .6                                ; CSIO at 6 X 8192 = 0C00h
    UNLOCK_N_MODIFY CSELIO
    
    clrf    TBLPTRU
    movlw   0c0h
    movwf   TBLPTRH
    movlw   0feh                            ; CS_CONFIG is at TBLPTRL = xxx1 1111
    movwf   TBLPTRL
    tblwt*+
    tblwt*
    
    movlw   0
    UNLOCK_N_MODIFY CSELIO
    return  


    
    END



